/*
 * ALSA driver for Panasonic UniPhier series.
 * 
 * Copyright (c) 2013 Panasonic corporation.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/spinlock.h>
#include <linux/vmalloc.h>
#include <linux/io.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/wait.h>
#include <linux/kthread.h>
#include <linux/delay.h>

#include "mn2ws-pcm.h"

MODULE_AUTHOR("Katsuhiro Suzuki <suzuki.katsuhiro002@jp.panasonic.com>");
MODULE_DESCRIPTION("Panasonic UniPhier PCM Capture Driver");
MODULE_LICENSE("GPL");

static void mn2ws_pcm_cap_transfer_sound(struct mn2ws_pcm_dev *d);

static int mn2ws_pcm_cap_open(struct snd_pcm_substream *substream);
static int mn2ws_pcm_cap_close(struct snd_pcm_substream *substream);
static int mn2ws_pcm_cap_ioctl(struct snd_pcm_substream *substream, 
	unsigned int cmd, void *arg);
static int mn2ws_pcm_cap_hw_params(struct snd_pcm_substream *substream, 
	struct snd_pcm_hw_params *hw_params);
static int mn2ws_pcm_cap_hw_free(struct snd_pcm_substream *substream);
static int mn2ws_pcm_cap_prepare(struct snd_pcm_substream *substream);
static int mn2ws_pcm_cap_trigger(struct snd_pcm_substream *substream, int cmd);
static snd_pcm_uframes_t mn2ws_pcm_cap_pointer(
	struct snd_pcm_substream *substream);
//static int mn2ws_pcm_cap_copy(struct snd_pcm_substream *substream, 
//	int channel, snd_pcm_uframes_t pos, void *buf, snd_pcm_uframes_t count);
//static int mn2ws_pcm_cap_silence(struct snd_pcm_substream *substream, 
//	int channel, snd_pcm_uframes_t pos, snd_pcm_uframes_t count);
static struct page *mn2ws_pcm_cap_page(struct snd_pcm_substream *subs, 
	unsigned long offset);
//static int mn2ws_pcm_cap_mmap(struct snd_pcm_substream *substream, 
//	struct vm_area_struct *vma);
//static int mn2ws_pcm_cap_ack(struct snd_pcm_substream *substream);

struct snd_pcm_ops mn2ws_pcm_capture_ops = {
	.open      = mn2ws_pcm_cap_open, 
	.close     = mn2ws_pcm_cap_close, 
	.ioctl     = mn2ws_pcm_cap_ioctl, 
	.hw_params = mn2ws_pcm_cap_hw_params, 
	.hw_free   = mn2ws_pcm_cap_hw_free, 
	.prepare   = mn2ws_pcm_cap_prepare, 
	.trigger   = mn2ws_pcm_cap_trigger, 
	.pointer   = mn2ws_pcm_cap_pointer, 
	//.copy      = mn2ws_pcm_cap_copy, 
	//.silence   = mn2ws_pcm_cap_silence, 
	.page      = mn2ws_pcm_cap_page, 
	//.mmap      = mn2ws_pcm_cap_mmap, 
	//.ack       = mn2ws_pcm_cap_ack, 
};

int mn2ws_pcm_cap_thread(void *data)
{
	struct mn2ws_pcm_dev *d = data;
	unsigned long hw_ptr_s, hw_ptr;
	struct timeval tv_s, tv_e, tv_i;
	int result = 0;
	
	if (d->cap.desc->init) {
		//init the DMA
		d->cap.desc->init(d);
	}
	
	while (!kthread_should_stop()) {
		//wait for start trigger
		spin_lock(&d->cap.spin);
		while (!d->cap.start_trigger) {
			spin_unlock(&d->cap.spin);
			
			if (kthread_should_stop()) {
				PRINTF_NOTE("ch%d_c: abort kthread\n", d->ch);
				result = 1;
				goto err_out1;
			}
			
			result = wait_event_interruptible(d->cap.q, 
				(d->cap.start_trigger
					|| kthread_should_stop()));
			if (result != 0) {
				PRINTF_WARN("ch%d_c: wait_event failed.\n", 
					d->ch);
				result = 1;
				goto err_out1;
			}
			
			spin_lock(&d->cap.spin);
		}
		d->cap.start_trigger = 0;
		
		spin_unlock(&d->cap.spin);
		
		
		//setup the DMA
		d->cap.desc->setup(d);
		
		mn2ws_pcm_dbg_pcmif_hwbuf(d, &d->cap);
		mn2ws_pcm_dbg_pcmif_ringbuf(d, &d->cap);
		
		//kick the DMA
		d->cap.desc->start(d);
		
		
		spin_lock(&d->cap.spin);
		//update HW read pointer
		if (d->cap.desc->set_devptr) {
			d->cap.desc->set_devptr(d, 
				get_rp_ringbuf(&d->cap.hw));
		}
		spin_unlock(&d->cap.spin);
		
		//wait for the HW write pointer
		do_gettimeofday(&tv_s);
		hw_ptr_s = hw_ptr = d->cap.desc->get_hwptr(d);
		while (hw_ptr == 0) {
			hw_ptr = d->cap.desc->get_hwptr(d);
		}
		do_gettimeofday(&tv_e);
		timersub(&tv_e, &tv_s, &tv_i);
		DPRINTF("ch%d_c: wait for HW %d.%06d[s]\n", 
			d->ch, (int)tv_i.tv_sec, (int)tv_i.tv_usec);
		
		//update HW buffer write pointer
		spin_lock(&d->cap.spin);
		set_wp_ringbuf(&d->cap.hw, d->cap.desc->get_hwptr(d));
		spin_unlock(&d->cap.spin);
		
		mn2ws_pcm_dbg_pcmif_ringbuf(d, &d->cap);
		
		//transfer sound data
		mn2ws_pcm_cap_transfer_sound(d);
		
		//stop the DMA
		d->cap.desc->stop(d);
		
		spin_lock(&d->cap.spin);
		d->cap.stop_trigger = 0;
		spin_unlock(&d->cap.spin);
	}
	
err_out1:
	if (d->cap.desc->term) {
		//term the DMA
		d->cap.desc->term(d);
	}
	
	return result;
}

static void mn2ws_pcm_cap_transfer_sound(struct mn2ws_pcm_dev *d)
{
	struct mn2ws_pcm_chip *chip;
	struct snd_pcm_substream *substream;
	struct snd_pcm_runtime *runtime;
	size_t remain;
	loff_t rp;
	size_t hw_remain, alsa_space, len, tran, writen;
	
	spin_lock(&d->cap.spin);
	
	//get ALSA PCM runtime
	chip = d->chip;
	substream = chip->substream;
	runtime = substream->runtime;
	
	spin_unlock(&d->cap.spin);
	
	while (!kthread_should_stop()) {
		spin_lock(&d->cap.spin);
		
		if (d->cap.stop_trigger) {
			spin_unlock(&d->cap.spin);
			break;
		}
		
		//get read pointer from bounce
		runtime = substream->runtime;
		if (runtime) {
			remain = d->cap.alsa.len - frames_to_bytes(runtime, 
				snd_pcm_capture_avail(runtime));
		} else {
			remain = 0;
		}
		remain = min(remain, d->cap.alsa.len - 1);
		
		rp = get_wp_ringbuf(&d->cap.alsa) + remain;
		set_rp_ringbuf(&d->cap.alsa, rp);
		
		//get write pointer from HW
		set_wp_ringbuf(&d->cap.hw, d->cap.desc->get_hwptr(d));
		
		//get space of ALSA buffer
		alsa_space = get_space_ringbuf(&d->cap.alsa);
		alsa_space -= alsa_space % 256;
		
		//get remain of HW buffer
		hw_remain = get_remain_ringbuf(&d->cap.hw);
		hw_remain -= hw_remain % 256;
		
		//transfer HW buffer -> bounce buffer
		len = min(alsa_space, hw_remain);
		while (len > 0 && !kthread_should_stop()) {
			tran = len;
			tran -= len % 256;
			
			writen = d->cap.desc->copy(d, &d->cap.hw, tran);
			if (writen == 0) {
				break;
			} else if (writen < 0) {
				PRINTF_WARN("ch%d_c: cannot copy to "
					"bounce buffer.\n", d->ch);
				break;
			}
			len -= writen;
		}
		
		spin_unlock(&d->cap.spin);
		
		//֤߰򹹿
		runtime = substream->runtime;
		if (runtime) {
			snd_pcm_period_elapsed(substream);
		}
		
		//wait event of HW
		d->cap.desc->wait_hwevent(d);
		
		mn2ws_pcm_dbg_pcmif_ringbuf_maxmin(d, &d->cap);
	}
}

static int mn2ws_pcm_cap_open(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	
	DPRINTF("%s\n", __func__);
	
	spin_lock(&d->cap.spin);
	
	//link chip -> substream
	chip->substream = substream;
	
	//determine the snd_pcm_hardware 
	memset(&runtime->hw, 0, sizeof(runtime->hw));
	d->cap.desc->hardware(d, &runtime->hw);
	
	spin_unlock(&d->cap.spin);
	
	return 0;
}

static int mn2ws_pcm_cap_close(struct snd_pcm_substream *substream)
{
	DPRINTF("%s\n", __func__);
	
	return 0;
}

static int mn2ws_pcm_cap_ioctl(struct snd_pcm_substream *substream, 
	unsigned int cmd, void *arg)
{
	int err = -ENOTTY;
	int result;
	
	//DPRINTF("%s %d\n", __func__, cmd);
	
	switch (cmd) {
	default:
		//default
		result = snd_pcm_lib_ioctl(substream, cmd, arg);
		break;
	}
	
	return result;
	
//err_out1:
	return err;
}

static int mn2ws_pcm_cap_hw_params(struct snd_pcm_substream *substream, 
	struct snd_pcm_hw_params *hw_params)
{
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	size_t size;
	int err;
	
	DPRINTF("%s\n", __func__);
	
	size = params_buffer_bytes(hw_params);
	
	err = mn2ws_pcm_pcmif_alloc_bounce_buffer(substream, size, &d->cap);
	if (err != 0) {
		PRINTF_WARN("ch%d_c: failed to allocate bounce buffer.\n", 
			d->ch);
		goto err_out1;
	}
	
	return 0;
	
err_out1:
	return err;
}

static int mn2ws_pcm_cap_hw_free(struct snd_pcm_substream *substream)
{
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	
	DPRINTF("%s\n", __func__);
	
	mn2ws_pcm_pcmif_free_bounce_buffer(substream, &d->cap);
	
	return 0;
}

static int mn2ws_pcm_cap_prepare(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	int err;
	
	DPRINTF("%s\n", __func__);
	
	runtime->delay = 0;
	
	mn2ws_pcm_dbg_pcmif_runtime(substream, &d->cap);
	
	//Reset HW ringbuf
	err = mn2ws_pcm_pcmif_reset_hw_ringbuf(substream, &d->cap);
	if (err != 0) {
		PRINTF_WARN("ch%d_c: failed to reset HW ring buffer.\n", d->ch);
		goto err_out1;
	}
	
	//Reset bounce ringbuf
	err = mn2ws_pcm_pcmif_reset_bounce_ringbuf(substream, &d->cap);
	if (err != 0) {
		PRINTF_WARN("ch%d_c: failed to reset bounce ring buffer.\n", 
			d->ch);
		goto err_out1;
	}
	
	return 0;
	
err_out1:
	return err;
}

/**
 * ꡼׶ػ
 */
static int mn2ws_pcm_cap_trigger(struct snd_pcm_substream *substream, int cmd)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	snd_pcm_uframes_t avail;
	int result = -EINVAL;
	
	DPRINTF("%s\n", __func__);
	
	avail = snd_pcm_capture_avail(runtime);
	DPRINTF("ch%d_c: snd_pcm_capture_avail: %d\n", d->ch, (int)avail);
	
	spin_lock(&d->cap.spin);
	
	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		DPRINTF("ch%d_c: start trigger\n", d->ch);
		
		d->cap.start_trigger = 1;
		result = 0;
		
		break;
	case SNDRV_PCM_TRIGGER_STOP:
		DPRINTF("ch%d_c: stop trigger\n", d->ch);
		
		d->cap.stop_trigger = 1;
		result = 0;
		
		break;
	default:
		//failed
		result = -EINVAL;
		break;
	}
	
	spin_unlock(&d->cap.spin);
	
	wake_up_interruptible(&d->cap.q);
	
	return result;
}

static snd_pcm_uframes_t mn2ws_pcm_cap_pointer(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	snd_pcm_uframes_t wr_frame;
	
	//DPRINTF("%s: avail:%d\n", __func__, 
	//	(int)snd_pcm_capture_avail(runtime));
	
	spin_lock(&d->cap.spin);
	
	//set read pointer of HW buffer
	if (d->cap.desc->set_devptr) {
		d->cap.desc->set_devptr(d, get_rp_ringbuf(&d->cap.hw));
	}
	
	//get write pointer of bounce buffer 
	wr_frame = bytes_to_frames(runtime, get_wp_ringbuf(&d->cap.alsa));
	
	spin_unlock(&d->cap.spin);
	
	//DPRINTF("pointer:%d\n", (int)wr_frame);
	
	return wr_frame;
}

static struct page *mn2ws_pcm_cap_page(struct snd_pcm_substream *substream, unsigned long offset)
{
	void *pageptr = substream->runtime->dma_area + offset;

	DPRINTF("%s\n", __func__);

	return vmalloc_to_page(pageptr);
}
